<?php
declare(strict_types=1);
namespace App\Integrations\OpenPass;

use App\Config;
use App\Support\DB;
use App\Support\Logger;

final class OpenPassClient
{
    private string $baseUrl;
    private string $username;
    private string $password;
    private string $applicationId;
    private string $deviceId;

    public function __construct()
    {
        $cfg = Config::openpass();
        $this->baseUrl = $cfg['base_url'];
        $this->username = $cfg['username'];
        $this->password = $cfg['password'];
        $this->applicationId = $cfg['application_id'];
        $this->deviceId = $cfg['device_id'];
    }

    /**
     * Gets a cached accessToken (and renews if expired).
     * Docs mention token from POST /login and then send with header 'accessToken'. 
     */
    public function accessToken(): string
    {
        $row = DB::one("SELECT token, expires_at FROM provider_tokens WHERE provider = 'openpass' LIMIT 1");
        if ($row && $row['token'] && $row['expires_at'] && strtotime($row['expires_at']) > time() + 60) {
            return $row['token'];
        }
        $tokenData = $this->login();
        $token = (string)($tokenData['accessToken'] ?? $tokenData['token'] ?? '');
        $expiresAt = $this->computeExpiry($tokenData);

        if ($token === '') {
            Logger::error('OpenPass login did not return token', ['response' => $tokenData]);
            throw new \RuntimeException('OpenPass token missing');
        }

        DB::exec("DELETE FROM provider_tokens WHERE provider='openpass'");
        DB::insert("INSERT INTO provider_tokens (provider, token, expires_at, created_at) VALUES ('openpass', ?, ?, NOW())", [
            $token, $expiresAt
        ]);

        return $token;
    }

    private function computeExpiry(array $tokenData): string
    {
        // If API provides expiresIn seconds use it; else default 30 minutes.
        $seconds = 1800;
        if (isset($tokenData['expiresIn']) && is_numeric($tokenData['expiresIn'])) {
            $seconds = (int)$tokenData['expiresIn'];
        }
        return gmdate('Y-m-d H:i:s', time() + max(60, $seconds));
    }

    public function login(): array
    {
        return $this->request('POST', '/login', [
            'username' => $this->username,
            'password' => $this->password,
        ], false);
    }

    /**
     * Generic request wrapper.
     * - Adds headers: accessToken + applicationId (+ deviceId if needed)
     * - Supports query parameters if $options['query'] passed.
     */
    public function request(string $method, string $path, array $body = [], bool $withAuth = true, array $options = []): array
    {
        $url = $this->baseUrl . $path;

        $headers = [
            'Content-Type: application/json',
            'Accept: application/json',
        ];

        // Many endpoints are secured with API Keys like applicationId / deviceId (docs examples).
        if ($this->applicationId !== '') $headers[] = 'applicationId: ' . $this->applicationId;
        if (!empty($options['sendDeviceIdHeader']) && $this->deviceId !== '') $headers[] = 'deviceId: ' . $this->deviceId;

        if ($withAuth) {
            $token = $this->accessToken();
            $headers[] = 'accessToken: ' . $token;
        }

        // Query string
        if (!empty($options['query']) && is_array($options['query'])) {
            $qs = http_build_query($options['query']);
            $url .= (str_contains($url, '?') ? '&' : '?') . $qs;
        }

        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, strtoupper($method));
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 20);
        curl_setopt($ch, CURLOPT_TIMEOUT, 60);

        if (in_array(strtoupper($method), ['POST','PUT','PATCH'], true)) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($body));
        }

        $raw = curl_exec($ch);
        $err = curl_error($ch);
        $code = (int) curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($raw === false) {
            Logger::error('OpenPass curl error', ['error' => $err, 'url' => $url]);
            throw new \RuntimeException('OpenPass network error');
        }

        $data = json_decode($raw, true);
        if (!is_array($data)) $data = ['raw' => $raw];

        if ($code >= 400) {
            Logger::error('OpenPass API error', ['status' => $code, 'url' => $url, 'response' => $data]);
        }

        return $data;
    }

    // Convenience methods you will map to actual endpoints based on your enabled products.

    public function createUser(array $payload): array {
        // TODO: replace endpoint path with the one in OpenPass docs for users
        return $this->request('POST', '/users', $payload, true);
    }

    public function createAccount(array $payload): array {
        // TODO: replace endpoint path with the one in OpenPass docs for account creation / CVU
        return $this->request('POST', '/accounts', $payload, true);
    }

    public function createTransfer(array $payload): array {
        // TODO: replace endpoint path with transfer endpoint in docs
        return $this->request('POST', '/transfers', $payload, true);
    }

    public function createPaymentLink(array $payload): array {
        // TODO: replace endpoint path with payment link endpoint in docs
        return $this->request('POST', '/payment-link', $payload, true, ['sendDeviceIdHeader' => true]);
    }
}
